function [T] = rhotran(rho,t)
% Subroutine to calculate the Resitivity Transform
% Function T given by equation 3.4.3 (page 36) of
% Koefoed, Geosounding Principles,1 - Resistivity
% Sounding Measurements, Elsevier, 1979
% rho is the vector of resistivities and t is the
% corresponding thickness of each layer.
% Note that the last layer is assumed to be the
% substratum (an semi-infinite halfspace).
% This program returns only a matrix T in which 
% the first column is the resistivity transform for the 
% uppermost layer (see page 34-35 of Koefoed).
% and the second column is the corresponding u = 1/lambda which
% corresponds to the offset. 
% The abscissae is log(u) and is in column 3. See Ghosh, D.P., Inversve filter 
% co-efficients for the computation
% of apparent resistivity standard curves for a horizontally
% stratified earth, Geophysical Prospecting, 19, 769-775, 1971).
% for a marginally better description than in Koefoed.

n = length(rho);
delty = (log(10))/6;  % O'Neill sampling for Ghosh's b filter.
y = -150*delty:delty:150*delty;  % y is the abscissa in 
				 % natural log space (log(AB/2))
u = exp(y);  m = length(u);  % u corresponds to metres 'offset'. % See Ghosh
           % This gives a range where T is = to rho(1) at the left and
           % = to rho(n) at the right but the limits are otherwise unrealistic

lambda = u.^(-1);

temp = zeros(m,n);

temp(:,n) = (rho(n)*1)*ones(m,1); % see 3.4.2 and 3.3.6

for i = n-1:-1:1
	r = rho(i); d = t(i);
	junk = (temp(:,i+1)+r*tanh(lambda'*d));
	temp(:,i) = junk./(1+temp(:,i+1).*tanh(lambda'*d)*(1/r));
end % i

T = u';   T(:,2) = temp(:,1); 


return;



