function [lossmatAPPROX, lossmatEXACT1, lossmatEXACT2] = runOB(method, fit, deltanu, n, N, nutoplot)

%set(groot, 'defaultFigureUnits', 'centimeters');

% Produces the output used in the paper

% Uses the 'parametric' or 'sequential' method (as in Stat&Computing paper)
% But only the mote stable sequential method is presented

% Set fit = 'linear', 'quadratic' or 'cubic'
% deltanu = distance between values of \nu
% n = design size
% N = size of design space
% When \nu = ntoplot, \xi_{\nu} is plotted

% To get the proper figure 1 and 2:
% [lossmatAPPROX, lossmatEXACT1, lossmatEXACT2] = runOB('sequential', 'linear', .02, 10, 40, .28);
% CB = .3305
% To get the proper figure 3:
% [lossmatAPPROX, lossmatEXACT1, lossmatEXACT2] = runOB('sequential', 'quadratic', .02, 14, 40, .6); 
% CB = .200 if nu = .51

close all
% This script changes all interpreters from tex to latex.
list_factory = fieldnames(get(groot,'factory'));
index_interpreter = find(contains(list_factory,'Interpreter'));
for i = 1:length(index_interpreter)
    default_name = strrep(list_factory{index_interpreter(i)},'factory','default');
    set(groot, default_name,'latex');
end

% Housekeeping
NUlength = 1+deltanu^-1;
NU = linspace(0,1,NUlength); %all values of nu

if  strcmp(fit, 'linear')
    x = linspace(-1, 1, N)';
    F = [ones(N,1), x];
    [Q,~] = qr(F,0);
    p = size(F,2);
    if strcmp(method, 'parametric')
        beta0 = [1;0];
        Lambda0 = rand*ones(p);
    end
elseif strcmp(fit, 'quadratic')
    x = linspace(-1, 1, N)';
    F = [ones(N,1), x, x.^2];
    [Q,~] = qr(F,0);
    p = size(F,2);
    if strcmp(method, 'parametric')
        beta0 = [1; 0; rand]; % rand(p,1);
        Lambda0 = eye(p);
    end
elseif strcmp(fit, 'cubic')
    x = linspace(-1, 1, N)';
    F = [ones(N,1), x, x.^2, x.^3];
    [Q,~] = qr(F,0);
    p = size(F,2);
    if strcmp(method, 'parametric')
        beta0 = [.91 0 .42 0]';
        Lambda0 = [ -0.86    0   -.06   0; ...
            0     .03   0   .17; ...
            -.06     0    .24    0; ...
            0    .17     0  .35];
    end
end

LOSSAPPROX  = zeros(NUlength,1);
LOSSVARAPPROX  = LOSSAPPROX ;
LOSSBIASAPPROX = LOSSAPPROX ;

LOSSEXACT1 = zeros(NUlength,1);
LOSSVAREXACT1= LOSSEXACT1;
LOSSBIASEXACT1 = LOSSEXACT1;

LOSSEXACT2 = zeros(NUlength,1);
LOSSVAREXACT2= LOSSEXACT2;
LOSSBIASEXACT2 = LOSSEXACT2;

lossmatAPPROX = [NU', LOSSAPPROX, LOSSVARAPPROX, LOSSBIASAPPROX];
lossmatEXACT1 = [NU', LOSSEXACT1, LOSSVAREXACT1, LOSSBIASEXACT1];
lossmatEXACT2 = [NU', LOSSEXACT2, LOSSVAREXACT2, LOSSBIASEXACT2];


if strcmp(method, 'sequential')
    delta0 = .02;
    for j = 1:NUlength
        nu = NU(j);
        % Initial design randomly chosen: size n0
        xi = zeros(N,1);
        xi(randsample(N,p)) = 1;
        xi = xi/sum(xi);
        [~, Inext, ~]  = summary(xi,  nu, Q); % current loss and next design point
        designsize = p;
        delta = 1;
        Istar = [];

        % Do the iterations:
        while (delta > delta0  && designsize < 3000) || designsize < 300
            istar = Inext;
            xi = (designsize*xi + [zeros((istar-1),1); 1; ...
                zeros(N-istar, 1)])/(designsize + 1);
            designsize = designsize + 1;
            [~, Inext, delta]  = summary(xi, nu, Q);
            Istar(designsize-p) = istar;           
        end

        % Delete the initial sample;
        % form the final design xistar from Istar alone

        Istar = Istar(101:length(Istar));
        nfinal = length(Istar);
        tab = tabulate(Istar);
        xistar = tab(:,2)/nfinal;

        % Symmetrize, and delete the tiniest design weights
        xistarsym = (xistar + flip(xistar))/2;
        % remove small weights, then renormalize:
        xistarsym(xistarsym < .05/n) = 0;
        xistarsym = xistarsym/sum(xistarsym);

        % Do the rounding, obtain the exact design of size n
        indices = (1:N)';
        xi0 = xistarsym;
        xi0exact1 = allocate1(n, xi0, indices);
        xi0exact2 = allocate2(n, N, xi0, indices, nu, deltanu, fit);

        [lossAPPROX, ~]  = summary(xi0, nu, Q);
        [lossEXACT1, ~]  = summary(xi0exact1, nu, Q);
        [lossEXACT2, ~]  = summary(xi0exact2, nu, Q);

        lossmatAPPROX(j,:) = [nu, lossAPPROX];
        lossmatEXACT1(j,:) = [nu, lossEXACT1];
        lossmatEXACT2(j,:) = [nu, lossEXACT2];

        if (nu==nutoplot)
            plotdesign(n, nu, x, xi0, xi0exact1, xi0exact2, ...
                lossAPPROX, lossEXACT1, lossEXACT2)
        end
    end

elseif strcmp(method, 'parametric')
    theta0 = [beta0; vecs(Lambda0, p)];
    for j = 1:NUlength
        nu = NU(j);
        if  nu == 0
            beta0 = [.6314 0 .7755 0]';
            Lambda0 = [-0.25         0    0.0128         0; ...
                0   -0.3227         0   -0.0751; ...
                0.0128         0   -0.2452         0; ...
                0   -0.0751         0    0.4667];
            %beta0 = beta0 + .1*randn(p,1);
            %Lambda0 = Lambda0 + .1*randn(p);
        end
        % Do the iterations:
        [theta0, ~, ~] = phi(theta0, Q, nu, p);
        fun = @(theta) lossonly(theta, Q, nu, p);
        nonlcon = @(theta) phicon(theta, Q, p);
        options = optimset('Display', 'none', 'algorithm', 'interior-point',...
            'TolFun', 1e-8, 'TolX', 1e-8, 'TolCon', 1e-8, ...
            'MaxIter', 1000, 'MaxFunEvals', 10000);
        % try 'interior-point' (default), then 'sqp', then'active-set' algorithm
        [theta, ~, ~, ~] = fmincon(fun,theta0,[],[],[],[],[],[], nonlcon, options);
        [~, xi0, ~] = phi(theta, Q, nu, p);

        % symmetrize:
        indices = (1:N)';
        xi0sym = (xi0 + flipud(xi0))/2;

        xi0 = xi0sym;
        xi0exact1 = allocate1(n, xi0, indices);
        xi0exact2 = allocate2(n, N, xi0, indices, nu, deltanu, fit);

        [lossAPPROX, ~, ~] = summary(xi0, nu, Q);
        [lossEXACT1, ~]  = summary(xi0exact1, nu, Q);
        [lossEXACT2, ~]  = summary(xi0exact2, nu, Q);

        lossmatAPPROX(j,:) = [nu, lossAPPROX];
        lossmatEXACT1(j,:) = [nu, lossEXACT1];
        lossmatEXACT2(j,:) = [nu, lossEXACT2];

        if (nu==nutoplot)
            plotdesign(n, nu, x, xi0, xi0exact1, xi0exact2, ...
                lossAPPROX, lossEXACT1, lossEXACT2);
        end
    end
end

LOSSAPPROX = lossmatAPPROX(:,2);
LOSSVARAPPROX  = lossmatAPPROX(:,3);
LOSSBIASAPPROX  = lossmatAPPROX(:,4);

LOSSEXACT1 = lossmatEXACT1(:,2);
LOSSVAREXACT1  = lossmatEXACT1(:,3);
LOSSBIASEXACT1  = lossmatEXACT1 (:,4);

LOSSEXACT2 = lossmatEXACT2(:,2);
LOSSVAREXACT2  = lossmatEXACT2(:,3);
LOSSBIASEXACT2  = lossmatEXACT2 (:,4);


f = figure(2);
f.Units = 'centimeters';
f.Position = [1, 1, 16, 5]; 
%figure(2)

hold on
subplot(1,2,1)
plot(NU, LOSSAPPROX, 'k', NU, LOSSVARAPPROX,'b-.', NU, LOSSBIASAPPROX, 'r--')
lgd = legend('$I_{\nu}(\xi_{\nu})$', '$s^{2}(\nu))$', ...
    '$b^{2}(\nu))$', 'fontsize', 10);
lgd.Location = "northwest";
lgd.Orientation = "vertical";
set(lgd,'EdgeColor','none');
xlabel({'$\nu$'; '(a) Continuous weights'})

subplot(1,2,2)
plot(NU, LOSSEXACT2, 'k', NU, LOSSVAREXACT2,'b-.', NU, LOSSBIASEXACT2, 'r--')
xlabel({'$\nu$'; '(b) Implementable weights'})
ylim([0 100])
hold off

f = figure(3);
f.Units = 'centimeters';
f.Position = [1, 1, 18, 11]; 
%figure(3)

hold on
subplot(2,3,4)
plot(NU, LOSSAPPROX, 'k', NU, LOSSVARAPPROX,'b-.', NU, LOSSBIASAPPROX, 'r--')
%ylim([0 80]);
lgd = legend('$I_{\nu}(\xi_{\nu})$', '$s^{2}(\nu))$', ...
    '$b^{2}(\nu))$', 'fontsize', 10);
lgd.Location = "northwest";
lgd.Orientation = "vertical";
set(lgd,'EdgeColor','none');
xlabel({'$\nu$'; '(d) Continuous weights'})

subplot(2,3,5)
plot(NU, LOSSEXACT2, 'k', NU, LOSSVAREXACT2,'b-.', NU, LOSSBIASEXACT2, 'r--')
xlabel({'$\nu$'; '(e) Implementable weights'})
ylim([0 100])

subplot(2,3,6)
plot(NU, LOSSEXACT1, 'k', NU, LOSSVAREXACT1,'b-.', NU, LOSSBIASEXACT1, 'r--')
xlabel({'$\nu$'; '(f) Pukelsheim-Rieder weights'})
ylim([0 100])

hold off



%% ============================= subfunctions =====================================================
%% ============================= subfunction "allocate1" =============================
% Pukelsheinm/Rieder approximation method; obeys sample size monotonicity
function xi = allocate1(n, xi, indices)
w = xi(xi>0); % postive design weights
supportindices = indices(xi>0);
el = length(supportindices);
frequency = ceil((n-el/2)*w);
discrepancy = sum(frequency) - n;

% alternate through 1:el:
El = (1:el)';
EL = [El flipud(El)];
EL = EL';
kk = EL(:);
kk = kk(1:el);
while ~(discrepancy == 0)
    for i = 1:el
        k = kk(i);
        if  discrepancy > 0  && (frequency(k) - 1)/w(k) == max((frequency - 1)./w)
            frequency(k) = frequency(k) - 1;
            discrepancy = sum(frequency) - n;
        end
        if  discrepancy < 0 && frequency(k)/w(k) == min(frequency./w)
            frequency(k) = frequency(k) + 1;
            discrepancy = sum(frequency) - n;
        end
    end
    discrepancy = sum(frequency) - n;
end
w = frequency/n;
xi(xi>0) = w;

%% ============================= subfunction "allocate2" =============================
% ad hoc approximation method; aims to preserve the value of the loss

function xi = allocate2(n, N, xi, indices, nu, deltanu, fit)
if (nu==1)
    nu = 1-deltanu/2;
end

x = linspace(-1, 1, N)';
supportindices = indices(xi>0);
x = x(supportindices);
N = length(supportindices);
if  strcmp(fit, 'linear')
    F = [ones(N,1), x];
    [Q,~] = qr(F,0);
elseif strcmp(fit, 'quadratic')
    F = [ones(N,1), x, x.^2];
    [Q,~] = qr(F,0);
elseif strcmp(fit, 'cubic')
    F = [ones(N,1), x, x.^2, x.^3];
    [Q,~] = qr(F,0);
end
w = xi(xi>0); % postive design weights
frequency = ceil(n*w);
discrepancy = sum(frequency) - n;
while ~(discrepancy == 0)
    M = length(frequency);
    D = diag(w);
    R = Q'*D*Q;
    S = Q'*(D^2)*Q;
    U = R\(R\S)';
    Tmat = T(nu, R, U, D, Q);
    dT = diag(Tmat);
    dT(frequency==0) = Inf;
    [~,loc] = min(dT);
    if frequency(loc) > 0
        frequency(loc) = frequency(loc)- 1;
        frequency(M+1-loc) = frequency(M+1-loc)- 1;
    end
    w = frequency/sum(frequency);
    frequency = ceil(n*w);
    discrepancy = sum(frequency) - n;
end
w = frequency/n;
xi(xi>0) = w;
xi = (xi + flip(xi))/2;


%% ============================= subfunction "lossonly" =============================
function loss = lossonly(theta, Q, nu, p)
[~,~, loss] = phi(theta, Q, nu, p);

%% ============================= subfunction "phi" =============================
% accepts a vector of parameters, returns (i) parameters meeting the
% constraints, (ii) design, (iii) loss, both evaluated at the new parameters

function [newtheta, xi, loss] = phi(theta, Q, nu, p)
beta = theta(1:p);
vec = theta((p+1):length(theta));
Lambda = unvecs(vec,p);
E = diag((-1).^(0:(p-1)));
beta = (beta + E*beta)/2;
Lambda = (Lambda + E*Lambda*E')/2;
xi = max(diag(Q*Lambda*Q'),0)./diag(Q*(beta*beta')*Q');

if all(xi==0)
    loss = 1e6;
    newtheta = theta + rand*ones(length(theta),1);
    N = length(xi);
    xi = ones(N,1)/N;
else
    I = sum(xi);
    beta = beta*sqrt(I); % now sum(xi) = 1
    Lambda = Lambda/(norm(beta)^2);
    beta = beta/norm(beta); % now norm(beta) = 1
    xi = max(diag(Q*Lambda*Q'),0)./diag(Q*(beta*beta')*Q');
    newtheta = [beta; vecs(Lambda, p)];
    [LOSS, ~, ~] = summary(xi, nu, Q);
    loss = LOSS(1);
end

%% ============================= subfunction "phicon" =============================
function [c,ceq] = phicon(theta, Q, p)
c = [];
beta = theta(1:p);
vec = theta((p+1):length(theta));
Lambda = unvecs(vec,p);
xi = max(diag(Q*Lambda*Q'),0)./diag(Q*(beta*beta')*Q');
E = diag((-1).^(0:(p-1)));
ceq = [sum(xi) - 1; norm(beta) - 1; ...
    norm(beta - E*beta); norm(Lambda - E*Lambda*E')];

%% ============================= subfunction "plotdesigns" =============================
function plotdesign(n, nu, x, xi0, xi0exact1, xi0exact2, ...
    lossAPPROX, lossEXACT1, lossEXACT2)

CB = sqrt(lossAPPROX(3)/lossAPPROX(2))

f = figure(1);
f.Units = 'centimeters';
f.Position = [1, 1, 16, 5]; 
%fontsize(11, "points")
subplot(1,2,1)
hold on

barwidth = .01;
str = {['$\nu$ = ', num2str(nu)]; ...
    ['$I_{\nu}(\xi_{\nu})$ = ', num2str(round(lossAPPROX(1),2))]; ...
    ['$s^{2}(\nu)$ = ', num2str(round(lossAPPROX(2),2))]; ...
    ['$b^{2}(\nu)$ = ', num2str(round(lossAPPROX(3),2))]};
xlim([-1.05 1.05]);
LIM = [0, 1.5*max(xi0)];
ylim(LIM);
bar(x, xi0, barwidth);
xlabel('(a) Continuous weights')
ylabel('design weights')
text(-.75, .8*LIM(2), str, 'fontsize', 10)
hold off

LIM = [0, max(2,n*LIM(2))];

subplot(1,2,2)
hold on
barwidth = .01;
str = {['$\nu$ = ', num2str(nu)]; ...
    ['$I_{\nu}(\xi_{\nu})$ = ', num2str(round(lossEXACT2(1),2))]; ...
    ['$s^{2}(\nu)$ = ', num2str(round(lossEXACT2(2),2))]; ...
    ['$b^{2}(\nu)$ = ', num2str(round(lossEXACT2(3),2))]};
xlim([-1.05 1.05]);
ylim(LIM);
yticks(0:floor(LIM(2)));
bar(x, n*xi0exact2 , barwidth);
xlabel('(b) Implementable weights')
ylabel('design allocations')
text(-.75, .8*LIM(2), str, 'fontsize', 10)
hold off


%f = figure(3);
%f.Units = 'centimeters';
%f.Position = [1, 1, 16, 11]; 
figure(3)

subplot(2,3,1)
hold on
barwidth = .01;
str = {['$\nu$ = ', num2str(nu)]; ...
    ['$I_{\nu}(\xi_{\nu})$ = ', num2str(round(lossAPPROX(1),2))]; ...
    ['$s^{2}(\nu)$ = ', num2str(round(lossAPPROX(2),2))]; ...
    ['$b^{2}(\nu)$ = ', num2str(round(lossAPPROX(3),2))]};
xlim([-1.05 1.05]);
LIM = [0, 1.5*max(xi0)];
ylim(LIM);
bar(x, xi0, barwidth);
xlabel('(a) Continuous weights')
ylabel('design weights')
text(-.75, .8*LIM(2), str, 'fontsize', 10)
hold off

LIM = [0, max(3,n*LIM(2))];

subplot(2,3,2)
hold on
barwidth = .01;
str = {['$\nu$ = ', num2str(nu)]; ...
    ['$I_{\nu}(\xi_{\nu})$ = ', num2str(round(lossEXACT2(1),2))]; ...
    ['$s^{2}(\nu)$ = ', num2str(round(lossEXACT2(2),2))]; ...
    ['$b^{2}(\nu)$ = ', num2str(round(lossEXACT2(3),2))]};
xlim([-1.05 1.05]);
ylim(LIM);
yticks(0:floor(LIM(2)));
bar(x, n*xi0exact2 , barwidth);
xlabel('(b) Implementable weights')
ylabel('design allocations')
text(-.75, .8*LIM(2), str, 'fontsize', 10)
hold off

subplot(2,3,3)
hold on
barwidth = .01;
str = {['$\nu$ = ', num2str(nu)]; ...
    ['$I_{\nu}(\xi_{\nu})$ = ', num2str(round(lossEXACT1(1),2))]; ...
    ['$s^{2}(\nu)$ = ', num2str(round(lossEXACT1(2),2))]; ...
    ['$b^{2}(\nu))$ = ', num2str(round(lossEXACT1(3),2))]};
xlim([-1.05 1.05]);
ylim(LIM);
yticks(0:floor(LIM(2)));
bar(x, n*xi0exact1 , barwidth);
xlabel('(c) Pukelsheim-Rieder weights')
ylabel('design allocations')
text(-.75, .8*LIM(2), str, 'fontsize', 10)
hold off




%% ============================= subfunction "summary" =============================
function [LOSS, Inext, delta]  = summary(xi, nu, Q)
% loss, next sequentially chosen obsevation and diagonal of the matrix T
D = diag(xi);
R = Q'*D*Q;
num = rcond(R);
if num < 1e-7 || isnan(num)
    disp(['nu = ', num2str(nu), '  rcond(R) = ', num2str(num)])
end
p = size(R,1);
S = Q'*(D^2)*Q;
U = R\(R\S)'; %%%%Rinv*S*Rinv';
[~, lambda] = eigs(U,1);
bias = lambda;
[~,E] = eigs(R);
var = sum(1./diag(E));
loss = (1-nu)*var + nu*bias;
LOSS = [loss, var, bias];
Tmat = T(nu, R, U, D, Q);
[d, Inext] = max(diag(Tmat));
delta = d/trace(D*Tmat) - 1;

%% ============================= subfunction "T" =============================
function out = T(nu, R, U, D, Q)
[z, lambda] = eigs(U,1);
vec = R\z;
B = 2*lambda*z*vec';
C = 2*(vec*vec');
mat = (R\Q')';
out = (1-nu)*(mat*mat') + nu*(Q*B*Q' - D*Q*C*Q');

% ============================= subfunction "unvecs" =============================
%Forms a matrix from the output of vecs
function mat = unvecs(vec, p)
mat = zeros(p);
for j = 1:p
    last = (j-1)*(2*p - j + 2)/2;
    mat(j:p,j) = vec((last + 1):(last + (p - j + 1)));
end
mat = mat + mat';
mat = mat - diag(diag(mat))/2;

%% ============================= subfunction "vecs" =============================
%Forms a vector from the lower triangle of a matrix
function vec = vecs(mat, p)
vec = [];
for j = 1:p
    vec = [vec; mat(j:p,j)];
end

